// ----------------------------------------------------------------
//  AN_OnMapHPGauge.js
//  Copyright (C) aonan 2016
// ----------------------------------------------------------------------------
//  
//  This software is released under the MIT License.
//  http://opensource.org/licenses/mit-license.php
//  
//  2016/**/** ver1.0 プラグイン公開
//  
// ----------------------------------------------------------------------------
/*:
 * @plugindesc マップ上HPゲージ
 * @author aonan
 * 
 * @param HideAtNewGame
 * @desc ニューゲーム時にゲージを非表示にします。(ONで非表示、OFFで表示、初期値はOFF)
 * @default OFF
 * 
 * @help
 * ----------------------------------------------------------------------------
 * マップ上HPゲージ by aonan
 * ----------------------------------------------------------------------------
 * 
 * アクションゲーム等に最適な、顔グラ付きHPゲージをマップ上に提供します。
 * 任意のマップ名表示機能付き(要プラグインコマンド実行)。
 * 
 * ※このプラグインはパーティーで行動しないゲーム(主人公一人のゲーム)を
 * 　想定しているため、パーティーのリーダー(先頭にいる人)以外には効果が
 * 　ありません。
 * 
 * ----------------------------------------------------------------------------
 * 
 * プラグイン コマンド：
 * 
 * 　OnMapHPGauge Show
 * 
 * 　　マップ上にHPゲージを表示します。
 * 　　（プラグイン パラメータの HideAtNewGame を設定している場合は、
 * 　　　これを実行するまで表示されません）
 * 
 * 　OnMapHPGauge Hide
 * 
 * 　　マップ上のHPゲージを隠します。
 * 
 * 　OnMapHPGauge MapName マップ名
 * 
 * 　　指定したマップ名をHPゲージの右空きスペースに表示します。
 * 　　（指定されたマップ名の中にあるアンダースコアは、空白に変換します。
 * 　　　全角であれば全角空白、逆に半角であれば半角空白になります。）
 * 
 * 　OnMapHPGauge MapName
 * 
 * 　　現在HPゲージの右空きスペースに表示しているマップ名を消します。
 * 
 * ----------------------------------------------------------------------------
 * Copyright (c) 2016 aonan
 *  
*/

var Imported = Imported || {};
Imported.AN_OnMapHPGauge = true;

(function() {
	'use strict';

	//----------------------------------------------------------------------------
	// プラグイン パラメーターの取得
	//
    var parameters = PluginManager.parameters('AN_OnMapHPGauge');
	var hideAtNewGame = (String(parameters['HideAtNewGame']).trim() || '').toUpperCase() === 'ON'; /* ON なら true、OFF なら false (但し余計な空白等無しで) */

	var windowStyle = 1;
	var gaugeWidth = 200;
	var gaugeHeight = 18;

	var actorFaceLeft = 0;
	var actorFaceTop = 0;
	var actorFaceWidth = 96;
	var actorFaceHeight = 96;

	var hpGaugeAreaTop = 376;
	var hpGaugeGroupTop = 72;
	var gaugeBorderColor = 'rgba(0,0,0,0.5)';
	var gaugeBorderWidth = 2;
	var gaugeWidthRate = 0.3;

	var noActorFace = false;
	var noBudget = false;
	var noMapName = false;

	var headerLeft_Offset = 4;
	var gaugeLeft_Offset = 54;
	var gaugeTop_Offset = 8;
	var budgetLeft_Offset = -68;
	var budgetWidth_Offset = 200;
	var mapNameLeft_Offset = 204;

	//----------------------------------------------------------------------------
	// 既定値またはプラグイン パラメーターから計算される情報 (パラメーターを直接は使用しない)
	//
	var hpMapName = '';
	var windowLeft = 0;
	var windowTop = 0;
	var windowWidth = SceneManager._screenWidth * 1.08; /* 長さは正確なのだが、わずかに届かない分の長さを追加するために1.08を乗算している */
	var windowHeight = SceneManager._screenHeight * 1.08; /* 長さは正確なのだが、わずかに届かない分の長さを追加するために1.08を乗算している */
	var hpGaugeAreaTop = windowHeight - (windowHeight * 0.23);
	var hpGaugeGroupTop = hpGaugeGroupTop;
	var hpHeaderLeft = headerLeft_Offset;
	var hpHeaderTop = hpGaugeAreaTop + hpGaugeGroupTop - 1;
	var hpGaugeLeft = hpHeaderLeft + gaugeLeft_Offset;
	var hpGaugeTop = hpGaugeAreaTop + hpGaugeGroupTop + gaugeTop_Offset;
	var hpGaugeWidth = windowWidth * gaugeWidthRate;
	var hpGaugeHeight = gaugeHeight;
	var hpBudgetLeft = hpGaugeLeft + hpGaugeWidth + (gaugeBorderWidth * 2) + budgetLeft_Offset;
	var hpBudgetTop = hpGaugeAreaTop + hpGaugeGroupTop;
	var hpMapNameLeft = hpBudgetLeft + mapNameLeft_Offset;
	var hpMapNameTop = hpGaugeAreaTop + hpGaugeGroupTop;

	//----------------------------------------------------------------------------
	// 内部用変数
	//
	var ignoreMask = true; /* ウィンドウのマスク処理を止めて重ね合わせを行うか */
	var windowVisible = true;
	var hideByWindowMessage = false; /* メッセージ画面が出ていたら隠す */
	var last_windowState = true;

	/* サブゲージ用設定 */
	var subGaugeVisible = false; /* サブゲージを表示するか */
	var subGaugeTitle = '';      /* サブゲージに表示するタイトル */
	var subGaugeX = 0;           /* サブゲージ横位置 */
	var subGaugeY = 0;           /* サブゲージ縦位置 */
	var subGaugeValue = 0;       /* サブゲージの現在値 */
	var subGaugeMaxValue = 0;    /* サブゲージの最大値 (除算で割る数に使用するため1にする) */

	//----------------------------------------------------------------------------
	// Game_Interpreter (プラグイン コマンドの実装、スクリプトで呼び出せる機能の追加)
	//
	var _AN_SUPERCLASS_Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
	Game_Interpreter.prototype.pluginCommand = function(command, args) {
		_AN_SUPERCLASS_Game_Interpreter_pluginCommand.call(this, command, args);
		if (command === 'OnMapHPGauge') {
			switch (args[0]) {
				case 'Show':
					windowVisible = true;
					break;
				case 'Hide':
					windowVisible = false;
					break;
				case 'MapName':
					if (args.length < 2) {
						hpMapName = ''; /* 第一引数 MapName の次、第二引数を指定しなかった場合はマップ名をクリアする */
					} else {
						hpMapName = args[1] || '';
						hpMapName = hpMapName.replace('＿','　'); /* 全角アンダースコアは全角空白へ置換 */
						hpMapName = hpMapName.replace('_',' '); /* 半角アンダースコアは半角空白へ置換 */
					}
					break;
				case 'SubGauge':
					switch (args[1]) {
						case 'Show':
							subGaugeVisible = true;
							break;
						case 'Hide':
							subGaugeVisible = false;
							break;
						case 'Setup':
							subGaugeTitle = args[2] || '';
							subGaugeTitle = subGaugeTitle.replace('＿','　'); /* 全角アンダースコアは全角空白へ置換 */
							subGaugeTitle = subGaugeTitle.replace('_',' '); /* 半角アンダースコアは半角空白へ置換 */
							subGaugeX = eval(args[3]) || 0;
							subGaugeY = eval(args[4]) || 0;
							subGaugeValue = eval(args[5]) || 0;
							subGaugeMaxValue = eval(args[6]) || 0;
							break;
						case 'Refresh':
							subGaugeValue = eval(args[2]) || 0;
							break;
					}
					break;
			}
		}
	};

	//----------------------------------------------------------------------------
	// DataManager (プラグインの情報をセーブデータにセーブ・ロード)
	//
	var _AN_SUPERCLASS_DataManager_makeSaveContents = DataManager.makeSaveContents;
	DataManager.makeSaveContents = function() {
		var contents = _AN_SUPERCLASS_DataManager_makeSaveContents();
		contents.windowVisible = windowVisible;
		contents.subGaugeVisible = subGaugeVisible;
		contents.subGaugeX = subGaugeX;
		contents.subGaugeY = subGaugeY;
		contents.subGaugeValue = subGaugeValue;
		contents.subGaugeMaxValue = subGaugeMaxValue;
		return contents;
	};
	var _AN_SUPERCLASS_DataManager_extractSaveContents = DataManager.extractSaveContents;
	DataManager.extractSaveContents = function(contents) {
		_AN_SUPERCLASS_DataManager_extractSaveContents(contents);
		windowVisible = contents.windowVisible;
		subGaugeVisible = contents.subGaugeVisible;
		subGaugeX = contents.subGaugeX;
		subGaugeY = contents.subGaugeY;
		subGaugeValue = contents.subGaugeValue;
		subGaugeMaxValue = contents.subGaugeMaxValue;
	};

	//----------------------------------------------------------------------------
	// Scene_Title (タイトル画面での処理)
	//
	var _AN_SUPERCLASS_scene_title_commandNewGame = Scene_Title.prototype.commandNewGame;
	Scene_Title.prototype.commandNewGame = function() {
		hpMapName = '';
		subGaugeVisible = false;
		subGaugeTitle = '';
		subGaugeX = 0;
		subGaugeY = 0;
		subGaugeValue = 0;
		subGaugeMaxValue = 0;
		if (hideAtNewGame === true) {
			windowVisible = false;
		}
		_AN_SUPERCLASS_scene_title_commandNewGame.call(this);
	};

	//----------------------------------------------------------------------------
	// Window_OnMapHPGauge (マップ上HPゲージの処理)
	//
	function Window_OnMapHPGauge() {
		this.initialize.apply(this, arguments);
	}

	Window_OnMapHPGauge.prototype = Object.create(Window_Base.prototype);
	Window_OnMapHPGauge.prototype.constructor = Window_OnMapHPGauge;

	Window_OnMapHPGauge.prototype.initialize = function(x, y, width, height) {
		Window_Base.prototype.initialize.call(this, 0, 0, windowWidth, windowHeight);
		this.refresh();
	};
    
	// ウィンドウのマスク処理をスキップしてウィンドウの重ね合わせを有効にする
	var _AN_SUPERCLASS_WindowLayer_webglMaskWindow = WindowLayer.prototype._webglMaskWindow;
	WindowLayer.prototype._webglMaskWindow = function(renderSession, win) {
		if (ignoreMask) return;
		_AN_SUPERCLASS_WindowLayer_webglMaskWindow.call(this, renderSession, win);
	};

	/* 描画内容の更新 */
	Window_OnMapHPGauge.prototype.refresh = function() {
		this.last_hp = !$gameParty.leader() ? 0 : $gameParty.leader().hp;
		this.last_leader = $gameParty.leader();
		this.last_windowVisible = this.isVisible();
		this.last_MapName = hpMapName;
		this.last_subGaugeVal = subGaugeValue;
		this.last_subGaugeVisible = subGaugeVisible;
		this.contents.clear();
		var actor = $gameParty.leader();
		if (windowStyle === 1) this.contents.fillRect(0, hpGaugeAreaTop + hpGaugeGroupTop, windowWidth, windowHeight, this.dimColor1());
		if (noActorFace === false) this.drawActorFaceScale(actor, actorFaceLeft, hpGaugeAreaTop + actorFaceTop, Window_Base._faceWidth, Window_Base._faceHeight, actorFaceWidth, actorFaceHeight);
		this.drawActorHp(actor, hpHeaderLeft, hpHeaderTop, hpGaugeLeft, hpGaugeTop, hpGaugeWidth, hpGaugeHeight, hpBudgetLeft, hpBudgetTop);
		if (noMapName === false) {
			this.changeTextColor(this.textColor(24));
			this.drawText(hpMapName, hpMapNameLeft, hpMapNameTop);
		}
		if (subGaugeVisible === true) {
			this.drawSubGauge(subGaugeX, subGaugeY, hpGaugeHeight);
		}
	};

	/* フレーム毎の更新処理 （更新条件も記述）*/
	Window_OnMapHPGauge.prototype.update = function() {
		Window_Base.prototype.update.call(this);
		/* メッセージ表示中の処理 */
		if ($gameMessage.isBusy()) {
			hideByWindowMessage = true;
		} else {
			hideByWindowMessage = false;
		}
		if ($gameParty.leader()) {
			if (this.last_hp != $gameParty.leader().hp) {
				this.refresh();
			}
		}
		if (this.last_leader != $gameParty.leader()) {
        	this.refresh();
    	}
		if (this.last_windowVisible != this.isVisible()) {
			this.refresh();
		}
		if (this.last_MapName != hpMapName) {
			this.refresh();
		}
		if (this.last_subGaugeVal != subGaugeValue) {
			this.refresh();
		}
		if (this.last_subGaugeVisible != subGaugeVisible) {
			this.refresh();
		}
	};

	/* ゲージ部分を開く */
	Window_OnMapHPGauge.prototype.open = function() {
		this.refresh();
		Window_Base.prototype.open.call(this);
	};

	/* ゲージ部分を閉じる */
	Window_OnMapHPGauge.prototype.updateClose = function() {
		if (this._closing) {
			this.openness -= 255;
			if (this.isClosed()) {
				this._closing = false;
			}
		}
	};
	
	/* アクター顔グラの描画処理 (拡大縮小に対応) */
	Window_OnMapHPGauge.prototype.drawActorFaceScale = function(actor, x, y, width, height, resizeWidth, resizeHeight) { /* アクター顔グラを横幅・縦幅指定で拡大縮小可能に */
		if (!actor) return;
		width = width || Window_Base._faceWidth;
		height = height || Window_Base._faceHeight;
		var bitmap = ImageManager.loadFace(actor.faceName());
		var pw = Window_Base._faceWidth;
		var ph = Window_Base._faceHeight;
		var sw = Math.min(width, pw);
		var sh = Math.min(height, ph);
		var dx = Math.floor(x + Math.max(width - pw, 0) / 2);
		var dy = Math.floor(y + Math.max(height - ph, 0) / 2);
		var sx = actor.faceIndex() % 4 * pw + (pw - sw) / 2;
		var sy = Math.floor(actor.faceIndex() / 4) * ph + (ph - sh) / 2;
		this.contents.blt(bitmap, sx, sy, sw, sh, dx, dy, resizeWidth, resizeHeight); /* bltの省略可能引数、dw と dh に横幅・縦幅を設定して拡大縮小可能に */
	};

	/*　HPゲージの描画 */
	Window_OnMapHPGauge.prototype.drawActorHp = function(actor, hX, hY, gX, gY, gWidth, gHeight, bX, bY) {
		if (!actor) return;
		gWidth = gWidth || 186;
		var color1 = this.hpGaugeColor1();
		var color2 = this.hpGaugeColor2();
		this.changeTextColor(this.normalColor());
		this.drawText(TextManager.hpA, hX, hY);
		this.drawGauge(gX, gY, gWidth, gHeight, actor.hpRate(), this.dimColor1(), this.gaugeBackColor(), color1, color2);
		if (noBudget === false) {
			if (Imported.YEP_CoreEngine) { /* YEP_CoreEngine.js プラグインが有効な場合は、HP残量表示の桁区切りを有効にする */
				this.drawCurrentAndMaxWithYEP(actor.hp, actor.mhp, bX, bY, budgetWidth_Offset, this.hpColor(actor), this.normalColor());
			} else {
				this.drawCurrentAndMax(actor.hp, actor.mhp, bX, bY, budgetWidth_Offset, this.hpColor(actor), this.normalColor());
			}
		}
	};

	/* サブゲージの描画 */
	Window_OnMapHPGauge.prototype.drawSubGauge = function(x, y, height, gColor1, gColor2) {
		var color1 = gColor1 || this.hpGaugeColor1();
		var color2 = gColor2 || this.hpGaugeColor2();
		this.changeTextColor(this.normalColor());
		this.drawText(subGaugeTitle, x, y);
		this.drawGauge(x + this.textWidth(subGaugeTitle), y + gaugeTop_Offset, 128, height, subGaugeMaxValue != 0 ? subGaugeValue / subGaugeMaxValue : 0, this.dimColor1(), this.gaugeBackColor(), color1, color2);
	};

	/* ゲージの描画処理 */
	Window_OnMapHPGauge.prototype.drawGauge = function(x, y, width, height, rate, borderColor, backColor, gaugeColor1, gaugeColor2) {
		var fillW = Math.floor(width * rate);
		if (gaugeBorderWidth > 0) {
			this.contents.fillRect(x - gaugeBorderWidth, y - gaugeBorderWidth, width + ( gaugeBorderWidth * 2 ), height + ( gaugeBorderWidth * 2 ), borderColor);
			this.contents.fillRect(x, y, width, height, backColor);
			this.contents.gradientFillRect(x, y, fillW, height, gaugeColor1, gaugeColor2);
		} else {
			this.contents.fillRect(x, y, width, height, backColor);
			this.contents.gradientFillRect(x, y, fillW, height, gaugeColor1, gaugeColor2);
		}
	};

	/* アクターのHP／最大HPの描画処理 (YEP_CoreEngine.js プラグインが有効な時) */
	Window_OnMapHPGauge.prototype.drawCurrentAndMaxWithYEP = function(current, max, x, y, width, color1, color2) {
		var labelWidth = this.textWidth(TextManager.hpA);
		var valueWidth = this.textWidth(Yanfly.Util.toGroup(max));
		var slashWidth = this.textWidth('/');
		var x1 = x + width - valueWidth;
		var x2 = x1 - slashWidth;
		var x3 = x2 - valueWidth;
		if (x3 >= x + labelWidth) {
			this.changeTextColor(color1);
			this.drawText(Yanfly.Util.toGroup(current), x3, y, valueWidth,
			'right');
			this.changeTextColor(color2);
			this.drawText('/', x2, y, slashWidth, 'right');
			this.drawText(Yanfly.Util.toGroup(max), x1, y, valueWidth, 'right');
		} else {
			this.changeTextColor(color1);
			this.drawText(Yanfly.Util.toGroup(current), x1, y, valueWidth,
			'right');
		}
	};

	/* アクターのHP／最大HPの描画処理 (通常時) */
	Window_OnMapHPGauge.prototype.drawCurrentAndMax = function(current, max, x, y, width, color1, color2) {
		var labelWidth = this.textWidth(TextManager.hpA);
		var valueWidth = this.textWidth('0000');
		var slashWidth = this.textWidth('/');
		var x1 = x + width - valueWidth;
		var x2 = x1 - slashWidth;
		var x3 = x2 - valueWidth;
		if (x3 >= x + labelWidth) {
			this.changeTextColor(color1);
			this.drawText(current, x3, y, valueWidth, 'right');
			this.changeTextColor(color2);
			this.drawText('/', x2, y, slashWidth, 'right');
			this.drawText(max, x1, y, valueWidth, 'right');
		} else {
			this.changeTextColor(color1);
			this.drawText(current, x1, y, valueWidth, 'right');
		}
	};

	/* ウィンドウの表示状態を取得*/
	Window_OnMapHPGauge.prototype.isVisible = function() {
		return ((windowVisible === true) && (hideByWindowMessage === false));
	};

	/* 他のプラグインからもアクセスできるようにエクスポート */
	window.Window_OnMapHPGauge = Window_OnMapHPGauge;

	//----------------------------------------------------------------------------
	// Scene_Map (マップのシーン管理)
	//

	/* マップ画面開始時の処理をオーバーライドして、HPゲージ等を表示する */
	var _AN_SUPERCLASS_Scene_Map_start = Scene_Map.prototype.start;
	Scene_Map.prototype.start = function() {
		_AN_SUPERCLASS_Scene_Map_start.call(this);
		if (this.HP_GaugeWindow.isVisible() === true) {
			this.HP_GaugeWindow.open();
		}
	};

	/* HPゲージを生成する処理 */
	Scene_Map.prototype.createOnMapHPGaugeWindow = function() {
		this.HP_GaugeWindow = new Window_OnMapHPGauge(this);
		this.HP_GaugeWindow.x = windowLeft;
		this.HP_GaugeWindow.y = windowTop;
		this.HP_GaugeWindow.width = windowWidth;
		this.HP_GaugeWindow.height = windowHeight;
		this.HP_GaugeWindow.padding = 0;
		this.HP_GaugeWindow.openness = 0;
		this.HP_GaugeWindow.setBackgroundType(2);
		this.addChild(this.HP_GaugeWindow);
	};

	/* 全ウィンドウを生成する処理 */
	var _AN_SUPERCLASS_Scene_Map_createAllWindows = Scene_Map.prototype.createAllWindows;
	Scene_Map.prototype.createAllWindows = function() {
		_AN_SUPERCLASS_Scene_Map_createAllWindows.call(this);
		this.createOnMapHPGaugeWindow(this);
	};

	/* マップ画面が更新された時の処理 */
	var _AN_SUPERCLASS_Scene_Map_update = Scene_Map.prototype.update;
	Scene_Map.prototype.update = function() {
		_AN_SUPERCLASS_Scene_Map_update.call(this);
		if (last_windowState != this.HP_GaugeWindow.isVisible()) {
			if (this.HP_GaugeWindow.isVisible() === false) {
				this.HP_GaugeWindow.close();
				last_windowState = false;
			} else if (this.HP_GaugeWindow.isVisible() === true) {
				this.HP_GaugeWindow.open();
				last_windowState = true;
			}
		}
	};

	/* シーンがマップ画面以外に遷移した時にウィンドウを見えなくする処理 */
	var _AN_SUPERCLASS_Scene_Map_stop = Scene_Map.prototype.stop;
	Scene_Map.prototype.stop = function() {
		this.HP_GaugeWindow.close();
		_AN_SUPERCLASS_Scene_Map_stop.call(this);
	};

	//----------------------------------------------------------------
	// Game_Event
	//
	Game_Event.prototype.moveTowardPlayerAlways = function() { /* 強化版自律移動「近づく」のスクリプト実装 */
		if ($gameMessage.isBusy()) return; /* メッセージ表示中の場合は移動を停止する */
		switch (Math.randomInt(8)) {
		case 0: case 1: case 2: case 3: case 4:
			this.moveTowardPlayer();
			break;
		case 5: 
			this.moveRandom();
			break;
		case 6: case 7: 
			this.moveForward();
			break;
		}
	};

})();